function [ zvec_expanded, muvec_expanded, orig_inds ] = Newton( f, muvec, Jac, constraint, tol, z_start )

% Newton is a code to find zvec(:,k) such that f(zvec(:,k),muvec(:,k))=0,
% k = 1,...,p: here muvec(:,k) shall be denoted
% the k^th "(continuation) data vector."

% Here f and Jac are handles to functions which evaluate f and the Jacobian of
% f, 'constraint' is a handle to a function which checks the constraints
% (true = satisfied), tol is the Newton residual tolerance, and z_start
% must satisfy f(z_start,muvec(:,1)) = 0 --- a good starting point.

numparvals = size(muvec,2); 
% Here numparvals = p is the number of data vectors in our trajectory.

orig_inds = logical(ones(1,numparvals));
% We will introduce intermediate data vectors to form an 
% expanded trajectory (or homotopy) muvec_expanded in order to ensure/improve 
% convergence of the Newton iteration.
% We denote by zvec_expanded the Newton solutions for the data vectors 
% of this expanded trajectory: f(zvec_expanded(:,k),muvec_expanded(:,k)) = 0
% for k = 1,...,p_expanded (the number of points in the expanded trajectory).
% The logical array orig_inds allows us to extract the p solutions zvec
% from the p_expanded solutions zvec_expanded: zvec = zvec_expanded(:,orig_inds).

numparvals_left = numparvals;
% Here numparvals_left is the number of data vectors in our expanded 
% trajectory which we must still visit.

ind_current = 1; 
% Here ind_current is the current index of the data vector in our expanded trajectory.

zvec_expanded = [];
muvec_expanded = muvec; % We begin with data vectors of interest as input to Newton.

z_iter = z_start; % Here z_iter is the current Newton iterate.


numNewtonitersmax_per = 6;
% If Newton has not converged for a particular data vector
% by numNewtonitersmax_per iterations, we insert a intermediate
% data vector --- in our expanded trajectory --- to ensure/improve convergence.
% (Recall the algorithm can not treat turning points or bifurcation points.)

while (numparvals_left > 0)
    % We visit each trajectory point and perform Newton iteration.
    
    mu_current = muvec_expanded(:,ind_current);
    % Here mu_current is the current data vector from our expanded trajectory. 
        
    numNewtoniters = 0;
    while ( norm(f(z_iter,mu_current)) >= tol ...
            & numNewtoniters < numNewtonitersmax_per )
        dz = -Jac(z_iter,mu_current)\f(z_iter,mu_current);
        z_iter = z_iter + dz;
        numNewtoniters = numNewtoniters + 1;
    end
   
    
    
    if ( norm(f(z_iter,mu_current)) <= tol & constraint(z_iter) )
        % If we have converged and our constraints on the solution are 
        % satisfied then we store the solution, we decrease the number
        % of data vectors still to visit, we increment the data vector
        % counter, and we set the initial guess for the next data vector in 
        % the trajectory to be the converged solution for the current data
        % vector.
        zvec_expanded = [zvec_expanded,z_iter];
        numparvals_left = numparvals_left - 1;
        ind_current = ind_current + 1;
        z_iter = zvec_expanded(:,end);
    else
        % If we have not converged, or our constraints are not satisfied,
        % we insert a new data vector "halfway" between our last data
        % vector in the expanded trajectory (perforce converged) and the 
        % current data vector in the expanded trajectory.
        % We also insert a false in orig_inds to indicate that
        % this new data vector is an intermediate data vector. 
        % Note since we have inserted a new point in muvec_expanded, 
        % we need to increment numparvals_left. In addition,  
        % we need to keep ind_current unchanged.
        % Finally, the initial guess for the new intermediate data
        % vector remains the last converged data vector.
        muvec_expanded = [muvec_expanded(:,1:ind_current-1),...
            0.5*(muvec_expanded(:,ind_current-1)+muvec_expanded(:,ind_current)),muvec_expanded(:,ind_current:end)];
        orig_inds = [orig_inds(1:ind_current-1),false,orig_inds(ind_current:end)];
        numparvals_left = numparvals_left+1;
        ind_current = ind_current;
        z_iter = zvec_expanded(:,end);
    end
end
    



end

