/*
 * 
 * Created on Apr 23, 2005
 * by ccfisch
 */

import java.io.*;
import java.util.regex.*;
import java.util.*;
import cern.colt.corejava.*;
import monte.DataPrinter;

/**
 * This class determines the autocorrelation function for a series of data (real-valued).
 * The input filename is provided and a file AutoCorr.out is output
 * When determining the autocorrelation function the the sequence (of length N) repeats itself
 * every N steps (just a convention). Thus, N should be much larger than the desired length.
 * @author ccfisch
 * Date: Apr 23, 2005
 *
 */
public class AutoCorr {

    static String m_filename;
    static int m_columnID;
    static int m_length;
    double[] m_averages;
    static String[] m_header;
    
    /**
     * 
     * @param args "input file" "column number" "length" 
     */
    public static void main(String[] args) {
        if(args.length<3){
            System.out.println("usage :  AutoCorr <filename> <column_number> <length>");
            System.out.println("output: AutoCorr.out contains a list of autocorrelation function values");
            return;
        }
        m_filename=args[0];
        m_columnID=(new Integer(args[1])).intValue();
        m_length=(new Integer(args[2])).intValue();
        try{
            double[] data = readData(m_filename,m_columnID);
            if((data.length/2)<m_length){
                System.out.println("specified length is too long; should be less than 1/2*num_points");
                return;
            }
            double[] avgs = detAverages(data);
            double[] vals = getAutoCorr(data,avgs,m_length);
            printFile(vals,m_header);
        }catch (Exception e){
            e.printStackTrace();
            System.out.println(e.getMessage());
        }
    }

    private static void printFile(double[] vals, String[] header) throws IOException{
        FileWriter fw = new FileWriter("AutoCorr.out");
        for(int i=0; i<header.length; i++) fw.write(header[i]);
        PrintfFormat d = new PrintfFormat("%14.8f");
        double sum = vals[0];
        for(int i=1; i<vals.length; i++) sum+=2.0*vals[i];
        fw.write("# 1+2.0*sum_1^N autocorr[i] = "+d.sprintf(sum)+DataPrinter.nl);
        for(int i=0; i<vals.length; i++) fw.write(d.sprintf(vals[i])+DataPrinter.nl);
        fw.close();
    }
    
    private static double[] getAutoCorr(double[] data, double[] avgs,int length){
        double[] vals = new double[length];
        int dlength = data.length;
        for(int i=0; i<length; i++){
            for(int j=0; j<dlength; j++){
                if((i+j)<dlength){
                    vals[i]+=data[j]*data[i+j];
                }else{
                    vals[i]+=data[j]*data[j+i-dlength];
                }
            }
            vals[i]/=dlength;
            vals[i]-=Math.pow(avgs[0],2.0);
            vals[i]/=(avgs[1]-avgs[0]*avgs[0]);
            vals[i]*=(1.0-(double)i/dlength);
        }
        return vals;
    }
    
    private static double[] readData(String filename, int column) throws IOException{
        LineNumberReader input = new LineNumberReader(new FileReader(m_filename));
        Pattern spc = Pattern.compile("\\s+");
        ArrayList header_lines = new ArrayList();
        ArrayList data = new ArrayList();
        String linein = input.readLine();
        linein=linein.trim();
        while(linein!=null){
            linein=linein.trim();
            if(Pattern.matches("^#.*",linein)){
                header_lines.add(linein);
                linein=input.readLine();
                linein=linein.trim();
                continue;
            }
            String[] cols = spc.split(linein);
            data.add(new Double(cols[column-1]));
            linein=input.readLine();
        }
        m_header=new String[header_lines.size()+1];
        for(int i=0; i<header_lines.size(); i++)  m_header[i]=(String)header_lines.get(i)+DataPrinter.nl;
        m_header[m_header.length-1]=new String("# column number "+column+DataPrinter.nl);
        double[] d = new double[data.size()];
        for(int i=0; i<data.size(); i++)  d[i]=((Double)data.get(i)).doubleValue();
        return d;
    }
    
    private static double[] detAverages(double[] data){
        double[] out = new double[2];
        int length = data.length;
        for(int i=0; i<length; i++){
            out[0]+=data[i];
            out[1]+=data[i]*data[i];
        }
        out[0]/=length;
        out[1]/=length;
        return out;
    }
    
}
