/*
 * monte
 * Created on Apr 20, 2005
 * by ccfisch
 */
package monte;

import cern.jet.random.engine.*;
import java.util.*;


/**
 * This is essentially a carbon copy of the two-dim lattice model code MC.f written
 * by Gerd Ceder in 1995 and updated by 3.320 TA's . The performance of this code is as-good
 * or better than g77-compiled MC.f
 * 
 * Some added functionality:
 * -keeps track of nearest and 2nd nearest neighbor correlation functions
 * -keeps track of (total magnetization/site)**2 and |total mag| for susceptibility
 * -the random number generator (MersenneTwister) should be better than the default rand()
 *  fortran subroutine and java.util.Random for that matter.
 * 
 * @author ccfisch
 * Date: Apr 23, 2005
 *
 */
public class MCSampler {
    final MersenneTwister rand = new MersenneTwister(new Date());
    final int SIZE;
    final int NUM_EQ;
    final int NUM_PASS;
    final double V1;
    final double V2;
    final double T;
    final double MU;
    final int[] modtable;
    int[][] config;
    InputReader m_params;
    
    /**
     * 
     * @param init_config
     * @param temp
     * @param mu
     * @param params
     */
    public MCSampler(int[][] init_config, double temp, double mu, InputReader params){
        m_params=params;
        SIZE    = params.m_SIZE;
        NUM_EQ  = params.m_NUM_EQ;
        NUM_PASS= params.m_NUM_PASS;
        V1   = params.m_V1;
        V2   = params.m_V2;
        T = temp;
        MU   = mu;
        config = new int[SIZE][SIZE];
        //copy initial configuration into new array
        for(int i=0; i<SIZE; i++){
            System.arraycopy(init_config[i],0,config[i],0,SIZE);
        }
        //make modtable....
        modtable = new int[SIZE+2];
        for(int i=-1; i<=SIZE; i++){
            //System.out.println("(i+1): "+(i+1)+" modtable: "+(i+SIZE)%SIZE);
            modtable[i+1]=(i+SIZE)%SIZE;            
        }
    }
    
    /**
     * performs a MC trajectory according to the parameters contained within
     * the InputReader object. 
     * @return
     */
    public MCData runMC(){
        //find total spin
        int total_spin=0;
        for(int i=0;i<SIZE;i++)
            for(int j=0; j<SIZE; j++)
                total_spin+=config[i][j];
        
        int tot_nn1 = getNN1(config);
        int tot_nn2 = getNN2(config);        
        double energy = getEnergy(config,V1,V2,MU);
        
        //make table of exponential function values for all single spin flips
        double[][][] explist = new double[9][9][3];
        for(int nn1=0;nn1<9;nn1+=2){
            for(int nn2=0; nn2<9; nn2+=2){
                for(int mui=0; mui<3; mui+=2){
                    explist[nn1][nn2][mui]=Math.exp(2.0*(V1*(nn1-4)+V2*(nn2-4)-MU*(mui-1))/T);
                    //System.out.println((nn1-4)+":"+(nn2-4)+":"+(mui-1)+" exp((E-mux)/kT)="+explist[nn1][nn2][mui]);
                }
            }
        }
        
        boolean print_allD=m_params.m_print_allD;
        double[] allE=null;
        double[] allS=null;
        double[] allNN1=null;
        double[] allNN2=null;
        if(print_allD){
            allE = new double[NUM_PASS-NUM_EQ];
            allS = new double[NUM_PASS-NUM_EQ];
            allNN1 = new double[NUM_PASS-NUM_EQ];
            allNN2 = new double[NUM_PASS-NUM_EQ];
        }
        double e_avg=0.0;
        double e2_avg=0.0;
        double spin_avg=0.0;
        double abs_spin_avg=0.0;
        double spin2_avg=0.0;
        double nn1_avg=0.0;
        double nn2_avg=0.0;
        int num_sites=SIZE*SIZE;
        int num_flips=0;
        int rindx=0;
        for(int i=0; i<NUM_PASS; i++){
            //perform a pass over the lattice...

            int num_pflips=0;                        
/*            for(int j=0; j<num_sites; j++){       //------------slower option----------
                  int x = rand.choose(0,SIZE-1);    //    picks sites at random 
                  int y = rand.choose(0,SIZE-1);    //---------------------------------
*/

            for(int x=0; x<SIZE; x++)               //------------faster (60%) option---                
                for(int y=0; y<SIZE; y++){          //     go through typewriter style
                                                    //EQ properties should not be changed by method of picking sites
                
                int nn1 = config[x][y]*(config[x][modtable[y+2]]+config[x][modtable[y]]+
                        config[modtable[x+2]][y]+config[modtable[x]][y]);
                
                int nn2 = config[x][y]*(config[modtable[x+2]][modtable[y+2]]+config[modtable[x+2]][modtable[y]]+
                        config[modtable[x]][modtable[y]]+config[modtable[x]][modtable[y+2]]);
                
                if(rand.nextFloat()<explist[nn1+4][nn2+4][config[x][y]+1]){
                //if(randlist[rindx++]<explist[nn1+4][nn2+4][config[x][y]+1]){ //use with cached RNG
                    //FLIP !!
                    total_spin+=-2*config[x][y];
                    tot_nn1+=-4*nn1;
                    tot_nn2+=-4*nn2;
                    energy+=-2.0*(nn1*V1+nn2*V2-MU*config[x][y]);
                    config[x][y]*=-1;
                    num_pflips++;
                }
                //if(rindx==randlist.length){rindx=0;}   //use with cached RNG
            }
            if(i>=NUM_EQ){
                num_flips+=num_pflips;
                e_avg+=energy/num_sites;
                e2_avg+=energy*energy/num_sites;
                spin_avg+=(double)total_spin/num_sites;
                abs_spin_avg+=(total_spin > 0) ? (double)total_spin/num_sites : (-1.0*total_spin)/num_sites;
                spin2_avg+=(double)total_spin*total_spin/num_sites;
                nn1_avg+=tot_nn1/(4.0*num_sites);
                nn2_avg+=tot_nn2/(4.0*num_sites);
                if(print_allD){
                    allE[i-NUM_EQ]=energy/num_sites;
                    allS[i-NUM_EQ]=(double)total_spin/num_sites;
                    allNN1[i-NUM_EQ]=tot_nn1/(4.0*num_sites);
                    allNN2[i-NUM_EQ]=tot_nn2/(4.0*num_sites);
                }
            }
        }
        e_avg/=(double)(NUM_PASS-NUM_EQ);
        e2_avg/=(double)(NUM_PASS-NUM_EQ);
        spin_avg/=(double)(NUM_PASS-NUM_EQ);
        abs_spin_avg/=(double)(NUM_PASS-NUM_EQ);
        spin2_avg/=(double)(NUM_PASS-NUM_EQ);
        nn1_avg/=(double)(NUM_PASS-NUM_EQ);
        nn2_avg/=(double)(NUM_PASS-NUM_EQ);
        double flip_freq=(double)num_flips/((NUM_PASS-NUM_EQ)*num_sites);
        
        if(m_params.m_print_allD){
            return new MCData(config,allE,allS,allNN1,allNN2,T,MU,flip_freq);
        }
        
        return new MCData(config,e_avg,e2_avg,spin_avg,spin2_avg,
                          nn1_avg,nn2_avg,T,MU,flip_freq,abs_spin_avg);
    }
    
    /**
     * returns energy of input configuration according to V1,V2, and MU
     * @param config
     * @param v1
     * @param v2
     * @param mu
     * @return
     */
    public static double getEnergy(int[][] config, double v1, double v2, double mu){
        int SIZE=config.length;
        double nn1=0;
        double nn2=0;
        double tot_spin=0;
        for(int i=0; i<SIZE; i++){
            for(int j=0; j<SIZE; j++){
                tot_spin+=config[i][j];
                nn1+=config[i][j]*(config[i][(j+1)%SIZE]+
                                   config[i][(j-1+SIZE)%SIZE]+
                                   config[(i+1)%SIZE][j]+
                                   config[(i-1+SIZE)%SIZE][j]);
                nn2+=config[i][j]*(config[(i+1)%SIZE][(j+1)%SIZE]+
                                   config[(i+1)%SIZE][(j-1+SIZE)%SIZE]+
                                   config[(i-1+SIZE)%SIZE][(j+1)%SIZE]+
                                   config[(i-1+SIZE)%SIZE][(j-1+SIZE)%SIZE]);
            }
        }
        //System.out.println("<nn1>="+nn1+" <nn2>="+nn2+" tot_spin="+tot_spin);
        double en=(0.5*(v1*nn1+v2*nn2)-mu*tot_spin);
        //System.out.println("en="+en);
        return en;
    }
    
    /**
     * returns the sum of all nearest-neighbor spin products
     * @param config
     * @return
     */
    public static int getNN1(int[][] config){
        int SIZE=config.length;
        int nn1=0;
        for(int i=0; i<SIZE; i++){
            for(int j=0; j<SIZE; j++){
                nn1+=config[i][j]*(config[i][(j+1)%SIZE]+
                        config[i][(j-1+SIZE)%SIZE]+
                        config[(i+1)%SIZE][j]+
                        config[(i-1+SIZE)%SIZE][j]);
            }
        }
        return nn1;
    }
    
    /**
     * returns the sum of all 2nd nearest neighbor spin products
     * @param config
     * @return
     */
    public static int getNN2(int[][] config){
        int SIZE=config.length;
        int nn2=0;
        for(int i=0; i<SIZE; i++){
            for(int j=0; j<SIZE; j++){
                nn2+=config[i][j]*(config[(i+1)%SIZE][(j+1)%SIZE]+
                        config[(i+1)%SIZE][(j-1+SIZE)%SIZE]+
                        config[(i-1+SIZE)%SIZE][(j+1)%SIZE]+
                        config[(i-1+SIZE)%SIZE][(j-1+SIZE)%SIZE]);
            }
        }
        return nn2;
    }
    
    
}
