# python implementation of "direct" gillespie method.
#
# lawrence david, 2006. ldavid@mit.edu

from gillespie_helper import *
import math
import random
import sys

# reaction parameters will be stored using dictionaries of
# dictionaries.  reactions parts are stored in three places: products
# live in product_dict, reactants live in substrate_dict, and
# propensities live in c_dict.  all three of these dictionaries are
# indexed by the same reaction names.
#
# examples of the dictionaries:
# consider RXN1: A + 2B -> C with propensity 0.01
#
# substrate_dict['RXN1'] = {'A':1, 'B':2}
# product_dict['RXN1'] = {'C':1}
# c_dict['RXN'] = 0.01
#
# counts_dict stores species' abundances.

substrate_dict, product_dict, c_dict = ReadInputFile(sys.argv[1])
counts_dict = ReadCountsFile(sys.argv[2])

# lists storing time-steps and species concentrations at each step
t_list = [0]
s_list = [counts_dict.copy()]

# how long a time period to simulate
t_max = 1000

# calculate both a_i and a
def GetA_i(substrate_dict,c_dict,counts_dict):

    a_i = {}
    a = 0

    # here, we first calculate h and then take product with c
    for rxn in substrate_dict:
        a_i_rxn = 1
        substrate_rxn = substrate_dict[rxn]

        # calculate h, the reaction propensity by successively taking
        # product of all species' counts involved in this reaction.
        # note that in most cases, substrate_rxn[species] is 1 and
        # NChooseK(n,1) just equals n.  
        for species in substrate_rxn:
            a_i_rxn *= NChooseK(counts_dict[species],substrate_rxn[species])

        a_i_rxn *= c_dict[rxn]
        a += a_i_rxn

        # store reaction probabilities in a dictionary
        a_i[rxn] = a_i_rxn

    return a_i, a

# calculate the initial a_i
a_i, a = GetA_i(substrate_dict,c_dict,counts_dict)

# the meat of the algorithm; here, we step through time, separately
# finding the time until the next reaction and the next occurring
# reaction.  
while t_list[-1] < t_max:

    # if there are no species left, the simuilation should end
    if a == 0:
        t.append(sys.maxint)
        break

    # how long until the next reaction
    r1 = random.random()
    tau = (1.0/a)*math.log(1.0/r1)

    # choose the next reaction by "binning" the interval (0,1); more
    # probable reactions are given larger bins.  what i'm doing is
    # picking a random number and then asking, "what is the smallest bin
    # boundary greater than my random number?"
    r2a = random.random()*a
    a_sum = 0
    for i in a_i:
        if r2a < (a_i[i]+a_sum):
            mu = i
            break
        a_sum += a_i[i]

    # update the substrate and product species counts involved
    for species in substrate_dict[mu]:
        counts_dict[species] -= substrate_dict[mu][species]
    for species in product_dict[mu]:
        counts_dict[species] += product_dict[mu][species]

    # record the events
    t_list.append(tau + t_list[-1])
    s_list.append(counts_dict.copy())

    # update a_i
    a_i, a = GetA_i(substrate_dict,c_dict,counts_dict)

# write the results to an output file
names = s_list[0].keys()
output_file = open(sys.argv[3],"w")
output_file.write("time ")
for k in s_list[0].keys():
    output_file.write("\t" + k)
output_file.write("\n")

# specify how many data points to use
data_points = int(sys.argv[4])
interval = len(t_list)/data_points

if interval < 1:
    interval = 1

for i in range(0,len(t_list),interval):
    output_file.write(str(t_list[i]))
    for j in names:
        output_file.write("\t" + str(s_list[i][j]))
    output_file.write("\n")
output_file.close()

