package ps5;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.WindowConstants;

// A simple GUI example. It contains multiple tabs, each of which displays a image
// file with something else drawn on it. The tabs use scrollbars in case the
// image is too large.

// This example also show how to use combo boxes and listeners that respond to
// user events.

// Note that there is no way to erase things that you draw on your picture.
// You can simulate that, however, by keeping a copy of your picture without
// the things that you drew on it.

public class GUIExample {
    // the image file we will use
    static String imageFile = "input/samples/picture.png";
    // the options for the combo box
    static String[] comboOptions = { "Option 1", "Option 2", "Option 3" };
    // a JFrame represents a window
    JFrame frame;
    // the comboBoxPane is the container of our combo box
    JPanel comboBoxPane;
    // the tabbedPane is the container of out tabs
    JTabbedPane tabbedPane;

    static Dimension frameSize = new Dimension(600,700);
    static Dimension tabSize = new Dimension(500,500);

    public GUIExample() throws Exception {
        // creates a window with the title Example
        frame = new JFrame("Example");
        // sets how to layout components within the window
        frame.getContentPane().setLayout(new BorderLayout());
        // creates the combo box, puts it in a container and adds to the window
        createComboBoxPane();
        // creates the tabs, puts them in a container and add it all to the window
        createTabPane();
        // when you close the window, the program exits
        frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        // arrange the components of the window nicely
        frame.pack();
        // set the size of the window. Note that if you pack the window after
        // this, the size will be changed.
        frame.setSize(frameSize);
        // displays the window on the screen
        frame.setVisible(true);
    }

    private void createComboBoxPane() {
        // this is the container for the combo box
        comboBoxPane = new JPanel();
        // this is the combo box with the options provided
        JComboBox comboBox = new JComboBox(comboOptions);
        // the first option is initially selected
        comboBox.setSelectedIndex(0);
        // the listener will respond when the user selects a new option
        comboBox.addActionListener(new ComboBoxListener(this));
        // adds the combo box to its container
        comboBoxPane.add(comboBox);
        // adds the container to the window, on the right (EAST)
        frame.getContentPane().add(comboBoxPane, BorderLayout.EAST);
    }

    private void createTabPane() throws Exception {
        // creates the container for the tabs
        tabbedPane = new JTabbedPane();
        // creates a listener that reponds when the user clicks on a picture
        PictureMouseListener listener = new PictureMouseListener(this);
        // creates a tab and returns its graphics. Graphics is something you
        // can draw on.
        Graphics graphics1 = createTab(imageFile, "Tab 1", listener);
        // draw with blue
        graphics1.setColor(Color.BLUE);
        // draw an oval shape
        graphics1.fillOval(10, 20, 40, 50);

        Graphics graphics2 = createTab(imageFile, "Tab 2", listener);
        graphics2.setColor(Color.RED);
        graphics2.drawLine(10, 10, 50, 50);
        // add the tab container to the window, on the left (WEST)
        frame.getContentPane().add(tabbedPane, BorderLayout.WEST);
    }

    private Graphics createTab(String imageName, String title,
            PictureMouseListener listener) throws Exception {
        // creates a container for the image
        JPanel picPane = new JPanel();
        // creates a scrollable container for the image. If the image is big,
        // scroll bars will appear
        JScrollPane scrollPane = new JScrollPane(picPane);
        // set the size of the picture container
        scrollPane.setPreferredSize(tabSize);
        // load the image file
        BufferedImage image = ImageIO.read(new File(imageName));
        // get its graphics object, so we can return it
        Graphics graphics = image.getGraphics();
        // make a label, which has the picture as its icon
        ImageIcon icon = new ImageIcon(image);
        JLabel label = new JLabel(icon);
        // add a listener that reponds to mouse clicks on the label
        label.addMouseListener(listener);
        // put the label in the image container
        picPane.add(label);
        // put the image container in the container for tabs
        tabbedPane.add(title, scrollPane);
        return graphics;
    }

    // this method will be called by the combo box listener
    public void optionSelected(String optionName) {
        // gets the index of the tab that is currently selected
        int index = tabbedPane.getSelectedIndex();
        // modifies the title of that tab
        tabbedPane.setTitleAt(index, optionName);
    }

    // this method will be called by the picture listener
    public void setSelectedTabTitle(String title) {
        int index = tabbedPane.getSelectedIndex();
        tabbedPane.setTitleAt(index, title);
    }

    // start our little application
    public static void main(String[] args) throws Exception {
        new GUIExample();
    }

    // the combo box listener responds to action events
    static class ComboBoxListener implements ActionListener {
        // this variable is necessary so that we callback
        private GUIExample example;
        public ComboBoxListener(GUIExample example) {
            this.example = example;
        }
        // this method is called automatically when an option is selected
        public void actionPerformed(ActionEvent e) {
            // retrieve the name of the option selected
            JComboBox cb = (JComboBox) e.getSource();
            String petName = (String) cb.getSelectedItem();
            // callback
            example.optionSelected(petName);
        }
    }

    // this listener responds to mouse events on the labels
    static class PictureMouseListener extends MouseAdapter {
        private GUIExample example;
        public PictureMouseListener(GUIExample example) {
            this.example = example;
        }
        // this method is called automatically when the user clicks on a label
        public void mouseClicked(MouseEvent e) {
            // retrieve the coordinates of the click, and callback
            example.setSelectedTabTitle(e.getX() + "," + e.getY());
        }
    }
}

