function [t,y,tsim,ysim] = sim_data(nval,pval,noclf)
% function [t,y,tsim,ysim] = sim_data(nval,pval,noclf)
%
% This function is similar to plot_data(), except that the user is
% prompted for information to describe a 2nd-order system (either
% the 2 pole locations, or zeta and wn), and a simulated step
% response for the defined 2nd-order system is then plotted on top
% of the raw scope data.
%
% ------------USAGE---------------------------------------------------
% MAKE SURE YOU HAVE SAVED DATA IN "SPREADSHEET" FORMAT!!! (.CSV file)
% --Inputs:
% nval is the number for the scope file. If your file
%    is saved as "TEK00014.CSV", then nval=14.
%    (nval=00014 is also fine!)
%    (nval='TEK00014.CSV' is ok, too, but be sure you are exact!)
% pval is an optional PLOTTING symbol, such as 'b.'
% noclf is an optional flag to PREVENT a clf command. (still must cla)
%    This is only useful if you are trying to create "subplots".
% 
% --Outputs:
% t is a 1xn vector of TIME values for the scope data
% y is a 1xn vector of VOLTAGE values for the scope data
% ysim is a 1xn vector which SIMULATES a STEP RESPONSE
% --------------------------------------------------------------------
% 
% --> TO RUN THIS FUNCTION to load and plot data from a scope data <-----
%     file called 'TEK00017.CSV', you would type the following:
% 
% [t,y,ysim] = plot_scope(17)

%---------CLF??----------------------------------------------
if ~exist('noclf'),    clf
else,    if noclf~=0,        clf,    end
end
%---------CREATE SCOPE DATA FILE NAME------------------------
if ischar(nval)
    sval=nval;
else
    sval='TEK';      % sval will be scope data file name
    n=num2str(nval);
    while length(n)<5
        n=['0' n];   % create 5-digit number
    end
    sval=[sval n '.CSV'];
end
%----------LOAD the file (a=load('TEK00014.DAT'))------------
q=fopen(sval,'r');  % first looks in local dir...
if q==-1
    sval=['a:' sval]; % ...then sets to a: if unsuccessful.
else
    fclose(q);
end
a=load(sval);
t=a(:,1);
y=a(:,2);
%---------PLOT DATA------------------------------------------
if ~exist('pval')
    pval='b.';
end
plot(t,y,'b.-');
xlabel('Time [s]'); ylabel('Raw scope data before scaling'); grid on
%---------GET INITIAL AND FINAL VALUES-----------------------
fprintf('Click on the initial (x,y) value of the step OR on a PEAK.\n')
title('Click on the initial (x,y) value of the step OR on a PEAK.')
pt1=ginput(1);
fprintf('Now click to indicate the y position of the final value of the step.\n');
fprintf('(Only the y position matters for this second point...)\n')
title('Click to indicate "final value"')
pt2=ginput(1);
t=t-pt1(1);
y=y-pt1(2);
A=pt2(2)-pt1(2);
y = (1/A)*y;
cla; plot(t,y,pval); grid on
title('Now go back to MATLAB window to continue...')
%---------PROMPT for 2nd-order system information------------
fprintf('To simulate a unit step response for a 2nd-order system,\n');
fprintf('I need two values from you.  They can be either:\n');
fprintf('  1 : s1 and s2 (the pole locations, by default), or\n');
fprintf('  2 : wd and zeta\n');
choice = input('Enter either 1 or 2: ');
choice = round(choice);
if choice == 2
    wd = input('Enter wd:  ');
    zeta = input('Enter zeta:  ');
    wn = wd*sqrt(1-zeta^2);
    s1 = -zeta*wn + i*wd;
    s2 = -zeta*wn - i*wd;
else
    if choice ~= 1
        fprintf('(default is POLE LOCATIONS...)')
    end
    fprintf('A pole may be either Real (-7) or Complex (-2 + 3.3*i)\n')
    s1 = input('Enter s1:  ');
    if imag(s1)~=0
        s2 = conj(s1);
        fprintf('Since s1 is complex --> s2 = %.3f\n',s2);
    else
        s2 = input('Enter second REAL pole, s2:  ');
    end
    wn = sqrt(s1*s2);
    zeta = -(s1+s2)/(2*wn);
end
%--------PLOT DATA AND THEORETICAL RESPONSE------------------
cla; plot(t,y,pval); hold on; grid on
if ~isempty(s2)
    k = (s1*s2);
    zpk1=zpk([],[s1 s2],k);
    [ysim,tsim]=step(zpk1,max(t)*[0:.001:1]);
    plot(tsim,ysim,'r--')
    if imag(s1)~=0
        zpk2=zpk([],[real(s1)],abs(real(s1)));
        [yenv,tenv]=step(zpk2,max(t)*[0:.001:1]);
        u1=plot(tenv,yenv,'g:')
        set(u1,'Color',[0 .6 0])
        u1=plot(tenv,2-yenv,'g:')
        set(u1,'Color',[0 .6 0])
        legend('normalized scope data','simulated step response','decay envelope')
    else
        legend('normalized scope data','simulated step response')
    end
    title(['Simulated Response is for s = ' [num2str(real(s1),'%.2f')] ...
            ' \pm ' [num2str(imag(s1),'%.2f')] 'j ; \omega_n = ' ...
            [num2str(wn,'%.2f')] ' [rad/s] ; \zeta = ' [num2str(zeta,'%.3f')]])
else
    k=abs(s1);
    zpk1=zpk([],[s1],k);
    [ysim,tsim]=step(zpk1,max(t)*[0:.001:1]);
    plot(tsim,ysim,'r--')
    legend('normalized scope data','simulated 1st-order step response')
    title(['Simulated Response is for s = ' [num2str(real(s1),'%.2f')]])
end
grid on; xlabel('Time(sec)'); ylabel('Normalized Output')

