# lecture3.py

import sys
sys.path.append('..')
from common.core import *
from common.gfxutil import *

from kivy.core.window import Window
from kivy.clock import Clock as kivyClock
from kivy.uix.label import Label
from kivy.graphics.instructions import InstructionGroup
from kivy.graphics import Color, Ellipse, Rectangle
from kivy.graphics import PushMatrix, PopMatrix, Translate, Scale, Rotate

from random import random, randint
import numpy as np


##############################################################################

# basics of getting player input:
# event-driven (touch_down, touch_up, touch_move)
# and polling/query (get_mouse_pos())
class MainWidget1(BaseWidget) :
    def __init__(self):
        super(MainWidget1, self).__init__()

        # create a label, keep track of it and add it so that it draws
        self.info = topleft_label()
        self.add_widget(self.info)

    def on_touch_down(self, touch) :
        print 'down', touch.pos

    def on_touch_up(self, touch) :
        print 'up', touch.pos

    def on_touch_move(self, touch) :
        print 'move', touch.pos

    # called every frame to update stuff (like the label)
    def on_update(self):
        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d' % kivyClock.get_fps()


##############################################################################

# Drawing a circle.
class MainWidget2(BaseWidget) :
    def __init__(self):
        super(MainWidget2, self).__init__()
        self.info = topleft_label()
        self.add_widget(self.info)

    def on_touch_down(self, touch) :

        # create a random color
        c = (random(), random(), random())
        clr = Color(*c)
        self.canvas.add(clr)

        # and a random radius:
        r = randint(10,60)
        p = touch.pos
        p = (touch.pos[0] - r, touch.pos[1] - r)

        elipse = Ellipse( pos=p, size=(r*2,r*2), segments = 40)
        self.canvas.add(elipse)


    def on_update(self):
        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d' % kivyClock.get_fps()
        self.info.text += "\nClick to draw"


##############################################################################

# keeping track of a canvas instruction
class MainWidget3(BaseWidget) :
    def __init__(self):
        super(MainWidget3, self).__init__()
        self.info = topleft_label()
        self.add_widget(self.info)

        self.color = Color(1,1,1)
        self.canvas.add(self.color)

        self.objects = []

    def on_touch_down(self, touch) :
        # and a random radius:
        r = randint(10,60)
        p = touch.pos
        obj = CEllipse( cpos=p, size=(r*2,r*2), segments = 20 )
        self.canvas.add(obj)
        self.objects.append(obj)


    def on_key_down(self, keycode, modifiers) :
        if keycode[1] == 'c':
            self.color.rgb = (random(), random(), random())

        # oops - should not do this:
        if keycode[1] == 'z':
            self.canvas.clear()

        # this is better:
        if keycode[1] == 'r':
            x = self.objects.pop()
            self.canvas.remove(x)

    def on_update(self):
        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d' % kivyClock.get_fps()
        self.info.text += "\nClick to draw"
        self.info.text += "\nc: change color"
        self.info.text += "\nz: clear canvas"
        self.info.text += "\nr: remove object"


##############################################################################
# KeyFrame Animation

# keeping track of a canvas instruction
class MainWidget4(BaseWidget) :
    def __init__(self):
        super(MainWidget4, self).__init__()
        self.info = topleft_label()
        self.add_widget(self.info)

        self.objects = []

    def on_touch_down(self, touch) :
        # initial settings
        p = touch.pos
        r = randint(10,60)
        c = (random(), random(), random())
        bubble = Bubble(p, r, c)
        self.canvas.add(bubble)
        self.objects.append(bubble)

    def on_update(self):
        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d' % kivyClock.get_fps()
        self.info.text += '\nbubbles:%d' % len(self.objects)

        dt = kivyClock.frametime

        kill_list = [b for b in self.objects if b.on_update(dt) == False]
        for b in kill_list:
            self.objects.remove(b)
            self.canvas.remove(b)


class Bubble(InstructionGroup):
    def __init__(self, pos, r, color):
        super(Bubble, self).__init__()

        center_x = Window.width/2
        center_y = Window.height/2

        self.radius_anim = KFAnim((0, r), (.1, 2*r), (3, 0))
        self.pos_anim    = KFAnim((0, pos[0], pos[1]), (3, center_x, center_y))

        self.color = Color(*color)
        self.add(self.color)

        self.circle = CEllipse(cpos = pos, size = (2*r, 2*r), segments = 40)
        self.add(self.circle)

        self.time = 0
        self.on_update(0)

    def on_update(self, dt):
        # animate radius
        rad = self.radius_anim.eval(self.time)
        self.circle.csize = (2*rad, 2*rad)

        # animate position
        pos = self.pos_anim.eval(self.time)
        self.circle.cpos = pos

        # advance time
        self.time += dt
        # continue flag
        return self.radius_anim.is_active(self.time)




##############################################################################
# Physics - based animation

gravity = np.array((0., -1800.))
damping = 0.9

class PhysBubble(InstructionGroup):
    def __init__(self, pos, r, color):
        super(PhysBubble, self).__init__()

        self.radius = r
        self.pos = np.array(pos, dtype=np.float64)
        self.vel = np.array((randint(-300, 300), 0), dtype=np.float64)

        self.color = Color(*color)
        self.add(self.color)

        self.circle = CEllipse(cpos=pos, csize=(2*r,2*r), segments = 40)
        self.add(self.circle)

        self.on_update(0)

    def on_update(self, dt):
        # integrate accel to get vel
        self.vel += gravity * dt

        # integrate vel to get pos
        self.pos += self.vel * dt

        # collision with floor
        if self.pos[1] - self.radius < 0:
            self.vel[1] = -self.vel[1] * damping
            self.pos[1] = self.radius

        self.circle.cpos = self.pos

        return True


class MainWidget5(BaseWidget) :
    def __init__(self):
        super(MainWidget5, self).__init__()
        self.info = topleft_label()
        self.add_widget(self.info)

        # AnimGroup handles drawing, animation, and object lifetime management
        self.objects = AnimGroup()
        self.canvas.add(self.objects)

    def on_touch_down(self, touch) :

        # initial settings
        p = touch.pos
        r = randint(10,60)
        c = (random(), random(), random())

        self.objects.add(PhysBubble(p, r, c))

    def on_update(self):
        self.objects.on_update()

        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d' % kivyClock.get_fps()
        self.info.text += '\nbubbles:%d' % self.objects.size()



class MainWidget6(BaseWidget) :
    def __init__(self):
        super(MainWidget6, self).__init__()
        self.info = topleft_label()
        self.add_widget(self.info)

        # dot showing origin
        self.canvas.add(Color(1,0,0, 0.5))
        self.canvas.add(Rectangle(pos=(0,0), size=(10,10)))

        # Rotate
        self.rotate = Rotate(angle = 0)
        self.canvas.add(self.rotate)

        # Translate
        self.canvas.add(Translate(100,100))

        # Rectangle object (with registration point)
        self.canvas.add(Color(0,1,0, 0.5))
        self.canvas.add(Rectangle(pos=(0,0), size=(10,10)))
        self.canvas.add(Color(0,0,1, 0.5))
        self.canvas.add(Rectangle(pos=(0,0), size=(200,100)))

    def on_touch_move(self, touch) :
        # apply rotation
        p = touch.pos
        self.rotate.angle = p[0]

    def on_update(self):
        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d\n' % kivyClock.get_fps()
        self.info.text += 'angle = %d' % self.rotate.angle





##############################################################################
# Reference Frames example

class Flower(InstructionGroup):
    def __init__(self, pos, num_petals, radius, color):
        super(Flower, self).__init__()

        self.add(Color(*color))
        self.add(PushMatrix())
        self.add(Translate(*pos))

        # make petals ellipses with these width and height:
        w = radius
        h = radius / num_petals ** .5

        # how much to rotate for each petal.
        d_theta =  360. / num_petals

        # use this Rotate to animate rotation for the whole flower
        self._angle = Rotate(angle = 0)
        self.add(self._angle)

        # use a loop with incremental rotation to place each petal at the proper
        # position and rotation
        for n in range(num_petals):
            self.add(Rotate(angle = d_theta))
            self.add(Translate(radius, 0))
            self.add(CEllipse(cpos = (0,0), csize = (w, h)))
            self.add(Translate(-radius, 0))

        self.add(PopMatrix())

    @property
    def angle(self):
        return self._angle.angle

    @angle.setter
    def angle(self, x):
        self._angle.angle = x


class MainWidget7(BaseWidget) :
    def __init__(self):
        super(MainWidget7, self).__init__()
        self.info = topleft_label()
        self.add_widget(self.info)

        self.flowers = []

        f = Flower((200, 300), 6, 80, (0, 1, 0))
        self.flowers.append(f)
        self.canvas.add(f)

        f = Flower((500, 400), 10, 100, (1, 0.7, 0.7))
        self.flowers.append(f)
        self.canvas.add(f)

    def on_touch_down(self, touch) :
        pass

    def on_update(self):
        self.info.text = str(self.get_mouse_pos())
        self.info.text += '\nfps:%d' % kivyClock.get_fps()

        for f in self.flowers:
            f.angle += 1

# pass in which MainWidget to run as a command-line arg
run(eval('MainWidget' + sys.argv[1]))
