package data;

import java.util.AbstractList;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class SpreadsheetFromText implements Spreadsheet
{
    public SpreadsheetFromText(String[] columnNames)
    {
        columns = new TypedColumn[columnNames.length];
        {
            int index = 0;
            for (String name : columnNames)
            {
                TypedColumn column = null;
                {
                    Matcher matcher = Pattern.compile("^(.*) \\(.*?\\)$")
                        .matcher(name);
                    if (matcher.matches())
                    {
                        final TypedColumn<Double> column_ =
                            new TypedColumn<Double>();
                        column_.type = Double.class;
                        column_.name = matcher.group(1);
                        column = column_;
                    }
                    else
                    {
                        final TypedColumn<Boolean> column_ =
                            new TypedColumn<Boolean>();
                        column_.type = Boolean.class;
                        column_.name = name;
                        column = column_;
                    }
                }
                column.index = index;
                column.fullName = name;
                columns[index] = column;
                columnIndexesByType.get(column.type).add(index);
                
                index++;
            }
        }
    }
    
    public List<Map<Column, Object>> rows()
    {
        return rows;
    }
    
    public <T> List<Column<T>> columnsByType(Class<T> type)
    {
        final List<Integer> columnIndexes = columnIndexesByType.get(type);
        return new AbstractList<Column<T>>()
        {
            public int size()
            {
                return columnIndexes.size();
            }
            
            public Column<T> get(int index)
            {
                return (Column<T>)columns[columnIndexes.get(index)];
            }
        };
    }
    
    public void addRow(String[] fieldStrings)
    {
        assert fieldStrings.length == columns.length;
        
        final Object[] fields = new Object[columns.length];
        {
            int columnIndex = 0;
            for (String fieldString : fieldStrings)
            {
                fields[columnIndex] = parseFieldFunctors.get(
                    columns[columnIndex].type).call(fieldString);
                columnIndex++;
            }
        }
        rows.add(new AbstractMap<Column, Object>()
        {
            public Set<Map.Entry<Column, Object>> entrySet()
            {
                return new AbstractSet<Map.Entry<Column, Object>>()
                {
                    public Iterator<Map.Entry<Column, Object>> iterator()
                    {
                        return new Iterator<Map.Entry<Column, Object>>()
                        {
                            public boolean hasNext()
                            {
                                return index < size();
                            }
                            
                            public Map.Entry<Column, Object> next()
                            {
                                final int index_ = index;
                                index++;
                                return new Map.Entry<Column, Object>()
                                {
                                    public Column getKey()
                                    {
                                        return columns[index_];
                                    }
                                    
                                    public Object getValue()
                                    {
                                        return fields[index_];
                                    }
                                    
                                    public Object setValue(Object value)
                                    {
                                        throw new
                                            UnsupportedOperationException();
                                    }
                                };
                            }
                            
                            public void remove()
                            {
                                throw new UnsupportedOperationException();
                            }
                            
                            protected int index;
                        };
                    }
                    
                    public int size()
                    {
                        return columns.length;
                    }
                };
            }
            
            public Object get(Object key)
            {
                return fields[((TypedColumn)key).index];
            }
        });
    }
    
    protected static class TypedColumn<T> extends IndexedColumn<T>
    {
        public String fullName()
        {
            return fullName;
        }
        
        protected String fullName;
        protected Class<T> type;
    }
    
    protected static abstract class ParseFieldFunctor
    {
        public abstract Object call(String valueString);
    }
    
    protected static final Map<Class, ParseFieldFunctor> parseFieldFunctors
        = new HashMap<Class, ParseFieldFunctor>();
    static
    {
        parseFieldFunctors.put(Boolean.class, new ParseFieldFunctor()
        {
            public Boolean call(String valueString)
            {
                return Integer.parseInt(valueString) != 0;
            }
        });
        parseFieldFunctors.put(Double.class, new ParseFieldFunctor()
        {
            public Double call(String valueString)
            {
                return Double.parseDouble(valueString);
            }
        });
    }
    
    protected TypedColumn[] columns = null;
    protected final Map<Class, List<Integer>> columnIndexesByType =
        new HashMap<Class, List<Integer>>();
    {
        for (Class type : parseFieldFunctors.keySet())
        {
            columnIndexesByType.put(type, new ArrayList<Integer>());
        }
    }
    protected final List<Map<Column, Object>> rows =
        new ArrayList<Map<Column, Object>>();
}
