package data;

import java.util.SortedMap;
import java.util.TreeMap;

import static java.lang.Math.max;
import static java.lang.Math.sqrt;
import static util.Math.ceil;
import static util.Math.floor;
import static util.Math.round;
import static util.Math.roundToSignificandResolution;
import static util.Math.roundToResolution;

public final class Statistics
{
    public static class ArrayStatistics
    {
        public double min;
        public double max;
        public double range;
        public double sum;
        public double average;
        public double sumOfSquares;
        public double stdDev;
        public SortedMap<Double, Integer> frequencies;
        public double separationMin;
        public double separationMax;
    }
    
    public static class HistogramParameters
    {
        public double min;
        public double max;
        public double binSize;
        public int binsCount;
        public boolean isDiscrete;
    }
    
    public static ArrayStatistics calculateStatistics(double[] data)
    {
        final ArrayStatistics returnValue = new ArrayStatistics();
        
        returnValue.min = Double.POSITIVE_INFINITY;
        returnValue.max = Double.NEGATIVE_INFINITY;
        returnValue.sum = 0.;
        returnValue.sumOfSquares = 0.;
        returnValue.frequencies = new TreeMap<Double, Integer>();
        for (double value : data)
        {
            if (value < returnValue.min)
            {
                returnValue.min = value;
            }
            if (value > returnValue.max)
            {
                returnValue.max = value;
            }
            returnValue.sum += value;
            returnValue.sumOfSquares += value * value;
            Integer frequency = returnValue.frequencies.get(value);
            if (frequency == null)
            {
                frequency = 0;
            }
            returnValue.frequencies.put(value, frequency + 1);
        }
        returnValue.range = returnValue.max - returnValue.min;
        returnValue.average = returnValue.sum / data.length;
        returnValue.stdDev = sqrt((returnValue.sumOfSquares / data.length)
            - (returnValue.average * returnValue.average));
        
        returnValue.separationMin = Double.POSITIVE_INFINITY;
        returnValue.separationMax = Double.NEGATIVE_INFINITY;
        {
            Double valuePrevious = null;
            for (Double value : returnValue.frequencies.keySet())
            {
                if (valuePrevious != null)
                {
                    final double separation = value - valuePrevious;
                    if (separation < returnValue.separationMin)
                    {
                        returnValue.separationMin = separation;
                    }
                    if (separation > returnValue.separationMax)
                    {
                        returnValue.separationMax = separation;
                    }
                }
                valuePrevious = value;
            }
        }
        
        return returnValue;
    }
    
    public static HistogramParameters configureHistogram(double[] data,
        int preferredBinsCount, double binSizeResolution)
    {
        final HistogramParameters returnValue = new HistogramParameters();
        final ArrayStatistics statistics = calculateStatistics(data);
        final double binSizeMin = roundToSignificandResolution(round,
            binSizeResolution, statistics.separationMin);
        returnValue.binSize = roundToSignificandResolution(round,
            binSizeResolution, statistics.range / preferredBinsCount);
        returnValue.isDiscrete = returnValue.binSize < binSizeMin;
        if (returnValue.isDiscrete)
        {
            returnValue.binSize = binSizeMin;
        }
        returnValue.min = roundToResolution(floor, returnValue.binSize,
            statistics.min);
        returnValue.max = roundToResolution(ceil, returnValue.binSize,
            statistics.max);
        if (returnValue.isDiscrete)
        {
            // Last bin needs to start, not end, with last discrete value.
            returnValue.max += binSizeMin;
        }
        returnValue.binsCount = (int)((returnValue.max - returnValue.min)
            / returnValue.binSize);
        return returnValue;
    }
}
